/****************************************************************************************************************************
  WebServerAP.ino - Simple Arduino web server sample for ESP8266 AT-command shield
  For ESP8266/ESP32 AT-command running shields

  ESP8266_AT_WebServer is a library for the ESP8266/ESP32 AT-command shields to run WebServer
  Based on and modified from ESP8266 https://github.com/esp8266/Arduino/releases
  Built by Khoi Hoang https://github.com/khoih-prog/ESP8266_AT_WebServer
  Licensed under MIT license
  
  Original author:
  @file       Esp8266WebServer.h
  @author     Ivan Grokhotkov
 *****************************************************************************************************************************/

// Credits of [Miguel Alexandre Wisintainer](https://github.com/tcpipchip) for this simple yet effective method
// For some STM32, there is only definition of Serial in variant.h, and is used for Serial/USB Debugging
// For example, in Nucleo-144 F767ZI original variant.h
//
// #define SERIAL_PORT_MONITOR     Serial
// #define SERIAL_PORT_HARDWARE    Serial
//
// To use ESP8266/ESP32-AT, we need another Serial, such as Serial1
// To do this, first, in corresponding variant.h, modify as follows:

// #define SERIAL_PORT_HARDWARE    Serial1
//
// then assign pins D0 = RX/D1 = TX to be Hardware Serial1 by putting in sketch as follows:
//
// #define EspSerial      SERIAL_PORT_HARDWARE    //Serial1
// HardwareSerial         Serial1(D0, D1);
//
// This must be included in defines.h for each board you'd like to use ESPSerial as Serial1
//
// The pin usage must be modified according to your boards.

#include "defines.h"

int status = WL_IDLE_STATUS;          // the Wifi radio's status
int reqCount = 0;                     // number of requests received

ESP8266_AT_WebServer server(80);

void printWifiStatus()
{
  // print the SSID/PASS of the network you're created:
  Serial.print(F("AP IP Address = "));
  Serial.println(WiFi.localIP());

  Serial.print(F("SSID: "));
  Serial.print(ssid);
  Serial.print(F(", PASS: "));
  Serial.println(pass);
}

void handleRoot()
{
#define BUFFER_SIZE     500
  
  char temp[BUFFER_SIZE];
  
  memset(temp, 0, sizeof(temp));
  
  int sec = millis() / 1000;
  int min = sec / 60;
  int hr = min / 60;
  int day = hr / 24;

  snprintf(temp, BUFFER_SIZE - 1,
           "<html>\
<head>\
<meta http-equiv='refresh' content='5'/>\
<title>%s</title>\
<style>\
body { background-color: #cccccc; font-family: Arial, Helvetica, Sans-Serif; Color: #000088; }\
</style>\
</head>\
<body>\
<h1>Hello from %s</h1>\
<h2>running WebServerAP</h2>\
<h3>on %s</h3>\
<h3>Uptime: %d d %02d:%02d:%02d</h3>\
<p>Requests received: %d</p>\
<p>Analog input A0: %ld</p>\
</body>\
</html>", BOARD_NAME, BOARD_NAME, SHIELD_TYPE, day, hr, min % 60, sec % 60, ++reqCount, analogRead(0));

  server.send(200, F("text/html"), temp);
}

void handleNotFound()
{ 
  String message = F("File Not Found\n\n");
  
  message += F("URI: ");
  message += server.uri();
  message += F("\nMethod: ");
  message += (server.method() == HTTP_GET) ? F("GET") : F("POST");
  message += F("\nArguments: ");
  message += server.args();
  message += F("\n");
  
  for (uint8_t i = 0; i < server.args(); i++)
  {
    message += " " + server.argName(i) + ": " + server.arg(i) + "\n";
  }
  
  server.send(404, F("text/plain"), message);
}

void setup()
{
  // Open serial communications and wait for port to open:
  Serial.begin(115200);
  while (!Serial);

  Serial.print(F("\nStarting WebServerAP on "));
  Serial.print(BOARD_NAME);
  Serial.print(F(" with "));
  Serial.println(SHIELD_TYPE); 
  Serial.println(ESP8266_AT_WEBSERVER_VERSION);

  // initialize serial for ESP module
  EspSerial.begin(115200);
  // initialize ESP module
  WiFi.init(&EspSerial);

  Serial.println(F("WiFi shield init done"));

  // check for the presence of the shield
  if (WiFi.status() == WL_NO_SHIELD)
  {
    Serial.println(F("WiFi shield not present"));
    // don't continue
    while (true);
  }

  Serial.print(F("Attempting to start AP with SSID = "));
  Serial.print(ssid);
  Serial.print(F(" and PW = "));
  Serial.println(pass);

  // uncomment these two lines if you want to set the IP address of the AP
  IPAddress localIp(192, 168, 120, 1);
  WiFi.configAP(localIp);

  // start access point
  int AP_channel = 1;
  status = WiFi.beginAP(ssid, AP_channel, pass, ENC_TYPE_WPA2_PSK);

  Serial.print(F("Access point started @ channel "));
  Serial.println(AP_channel);

  printWifiStatus();

  server.on(F("/"), handleRoot);
  
  server.on(F("/inline"), []()
  {
    server.send(200, F("text/plain"), F("This works as well"));
  });

  server.onNotFound(handleNotFound);

  // start the web server on port 80
  server.begin();

  // print your WiFi shield's IP address:
  IPAddress ip = WiFi.localIP();
  
  Serial.print(F("WebServer started @ "));
  Serial.println(ip);
  
  // print where to go in a browser:
  Serial.print(F("To see this page in action, open a browser to http://"));
  Serial.println(ip);
}


void loop()
{
  server.handleClient();
}
